/* (c) 1999-2000 Tino Schwarze, see COPYING for details */
/**@pkg cObject.cVisibleObject.cInteractiveObject.cPlane*/
/**
 * plane derived from class cInteractiveObject
 *
 * #include "cPlane.hh"
 * -lglut
 *
 * @see cInteractiveObject
 * @pkgdoc cObject.cVisibleObject.cInteractiveObject.cPlane
 */

#ifndef cPlane_hh
#define cPlane_hh

#include "cInteractiveObject.hh"

/**
 * Simple plane object.
 	*
	* This object displays a simple plane, either solid or wireframed.
	* Further, a display list is used to draw the plane.
	* The plane starts at (-size_x/2,-size_y/2,0) and extends to
	* (size_x/2, size_y/2, 0)
	* Texture coordinates are generated also. (default: (0.0,0.0)-(1.0,1.0))
	*/
class cPlane 
:	public cInteractiveObject
{
// LIFECYCLE
public:
	/**
	 * an enum to describe the plane's type (solid or wireframed)
	 	*/
	typedef enum ePlaneType {
		PLANE_SOLID,
		PLANE_WIREFRAMED
	} tePlaneType;

	/**
	 * default constructor w/ optional object name
	 	* @param name name of object (optional)
	 	*/
	cPlane (
		cEventDispatcher *disp,
		const char *name = NULL);

	/**
	 * constructor w/ optional size and object name
	 	* @param size_x (GLdouble) x side length of plane (default: 1.0)
	 	* @param size_y (GLdouble) y side length of plane (default: 1.0)
	 	* @param name name of object (optional)
	 	*/
	cPlane (
		cEventDispatcher *disp,
		GLdouble size_x, 
		GLdouble size_y, 
		const char *name = NULL);

	/**
	 * default constructor
	 	* @param size_x (GLdouble) x side length of plane (default: 1.0)
	 	* @param size_y (GLdouble) y side length of plane (default: 1.0)
	 	* @param solid (tePlaneType) draw solid or wireframe plane (default: solid)
		* @param name name of object (optional)
	 	*/
	cPlane (
		cEventDispatcher *disp,
		GLdouble size_x, 
		GLdouble size_y, 
		tePlaneType solid,
		const char *name = NULL);

	/**
	 * default constructor
	 	* @param size_x (GLdouble) x side length of plane (default: 1.0)
	 	* @param size_y (GLdouble) y side length of plane (default: 1.0)
	 	* @param solid (tePlaneType) draw solid or wireframe plane (default: solid)
		* @param resolution subdivisions per side
		* @param name name of object (optional)
	 	*/
	cPlane (
		cEventDispatcher *disp,
		GLdouble size_x, 
		GLdouble size_y, 
		tePlaneType solid,
		int resolution,
		const char *name = NULL);

	/**
	 * destructor
	 	*/
	virtual ~cPlane ();

	/**
	 * initialization function
	 	*/
	virtual int Init ();

	/**
	 * set texture coordinates to use
	 	* @param x_min from-x
	 	* @param x_max to-x
		* @param y_min from-y
	 	* @param y_max to-y
	 	*/
	virtual void SetTextureDomain (
		GLfloat x_min, GLfloat x_max,
		GLfloat y_min, GLfloat y_max);

protected:	
	/**
	 * drawing function
	 	*/
	virtual void DrawThisObject ();

	/**
	 * return class name
	 	*/
	virtual const char *GetDefaultName () const
	{
		return "cPlane";
	}

	/**
	 * create solid plane at given resolution
	 	*/
	virtual void MakeSolidPlane ();

	
	/**
	 * create wire framed plane at given resolution
	 	*/
	virtual void MakeWirePlane ();

	/** ID of display list which contains the plane */
	GLuint mDisplayList;

	/** Is the plane solid or wireframed? */
	tePlaneType mSolid;

	/** side length of plane */
	GLdouble mSizeX;
	GLdouble mSizeY;

	/** resolution of plane, if given */
	int mResolution;

	/** texture domain to use */
	GLfloat mTexFromX, mTexToX, mTexFromY, mTexToY;
};

#endif	// ifndef cPlane_hh
