/* (c) 1999-2000 Tino Schwarze, see COPYING for details */
/**
 * implementation of class cPlane
 *
 * -lglut
	*/

// get glut library functions
#include <GL/glut.h>

#include "cPlane.hh"

/**
 * default constructor w/ optional object name
 	* @param name name of object (optional)
	*/
cPlane::cPlane (
	cEventDispatcher *disp,
	const char *name)
: 	cInteractiveObject (disp, name),
  	mDisplayList (0),
  	mSolid (PLANE_SOLID),
	mSizeX (1.0),
	mSizeY (1.0),
	mResolution(1),
	mTexFromX (0.0), mTexToX (1.0),
	mTexFromY (0.0), mTexToY (1.0)
{
	ENTER_OBJECT_METHOD("cPlane::cPlane (const char *)");
}


/**
 * constructor w/ optional size and object name
 	* @param size_x x size of plane (default: 1.0)
 	* @param size_y y size of plane (default: 1.0)
 	* @param name name of object (optional)
	*/
cPlane::cPlane (
	cEventDispatcher *disp,
	GLdouble size_x, 
	GLdouble size_y, 
	const char *name)
:	cInteractiveObject (disp, name),
	mDisplayList (0),
	mSolid (PLANE_SOLID),
	mSizeX (size_x),
	mSizeY (size_y),
	mResolution (1),
	mTexFromX (0.0), mTexToX (1.0),
	mTexFromY (0.0), mTexToY (1.0)
{
	ENTER_OBJECT_METHOD("cPlane::cPlane (GLdouble, const char *)");
}


/**
 * constructor w/ initialization of size, solid and name
 	* @param size_x x size of plane (default: 1.0)
 	* @param size_y y size of plane (default: 1.0)
	* @param solid solid or wire frame? (default: solid)
	* @param name name of object (optional)
 	*/
cPlane::cPlane (
	cEventDispatcher *disp,
	GLdouble size_x, 
	GLdouble size_y, 
	tePlaneType solid,
	const char *name)
: 	cInteractiveObject (disp, name), 
  	mDisplayList(0),
  	mSolid (solid), 
	mSizeX (size_x),
	mSizeY (size_y),
  	mResolution (1),
	mTexFromX (0.0), mTexToX (1.0),
	mTexFromY (0.0), mTexToY (1.0)
{
	ENTER_OBJECT_METHOD ("cPlane::cPlane (GLdouble size, GLboolean solid)");
	// this is enough (mDisplayList is aquired during Init()
}

/**
 * constructor w/ initialization of size, solid and name
 	* @param size_x x size of plane (default: 1.0)
 	* @param size_y y size of plane (default: 1.0)
	* @param solid solid or wire frame? (default: solid)
	* @param resolution subdivisions per side
	* @param name name of object (optional)
 	*/
cPlane::cPlane (
	cEventDispatcher *disp,
	GLdouble size_x, 
	GLdouble size_y, 
	tePlaneType solid,
	int resolution,
	const char *name)
: 	cInteractiveObject (disp, name), 
  	mDisplayList(0),
  	mSolid (solid), 
	mSizeX (size_x),
	mSizeY (size_y),
  	mResolution (resolution),
	mTexFromX (0.0), mTexToX (1.0),
	mTexFromY (0.0), mTexToY (1.0)
{
	ENTER_OBJECT_METHOD ("cPlane::cPlane (GLdouble size, GLboolean solid)");
	// this is enough (mDisplayList is aquired during Init()
}
/**
 * destructor
 	*/
cPlane::~cPlane ()
{
	ENTER_OBJECT_METHOD ("cPlane::~cPlane ()");
	if (mDisplayList != 0)
		glDeleteLists (mDisplayList, 1);
}

/**
 * initialization function (aquires display list and compiles it)
 	*/
int cPlane::Init ()
{
	ENTER_OBJECT_METHOD ("cPlane::Init ()");

	// have at least one quad
	if (mResolution < 1)
		mResolution = 1;

	// generate one display list
	mDisplayList = glGenLists (1);

	if (mDisplayList == 0)
		return -1;

	// now fill some data into the list, but do not display anything
	// (no error checking possible there)
	glNewList (mDisplayList, GL_COMPILE);

	if (mSolid == PLANE_SOLID)
	{
		MakeSolidPlane ();
	}
	else
	{
		MakeWirePlane ();
	}
	
	glEndList ();	// complete display list

	// call inherited Init (cumpulsory!) after we're set up
	cInteractiveObject::Init ();

	return 0;	// that's all folks - success.
}

void cPlane::SetTextureDomain (
	GLfloat x_min, GLfloat x_max,
	GLfloat y_min, GLfloat y_max)
{
	mTexFromX = x_min;
	mTexToX = x_max;
	mTexFromY = y_min;
	mTexToY = y_max;
}

void cPlane::DrawThisObject ()
{
	ENTER_OBJECT_METHOD ("cPlane::DrawThisObject ()");
	glCallList (mDisplayList);
}

/**
 * create solid plane
 	*/
void cPlane::MakeSolidPlane ()
{
	const GLfloat step_x = (mSizeX/(mResolution));
	const GLfloat step_y = (mSizeY/(mResolution));
	const GLfloat step_tx = (mTexToX - mTexFromX)/(mResolution);
	const GLfloat step_ty = (mTexToY - mTexFromY)/(mResolution);

	const GLfloat start_x = -mSizeX/2.0;

	GLfloat cur_x, cur_tx;
	GLfloat cur_y = -mSizeY/2.0, cur_ty = mTexFromY;

	// normal points "up" in Z direction
	glNormal3f (0.0, 0.0, 1.0);

	for (int y = 0; y < mResolution; y++)
	{
		const GLfloat next_y = cur_y + step_y;
		const GLfloat next_ty = cur_ty + step_ty;

		glBegin (GL_TRIANGLE_STRIP);

		cur_x = start_x;
		cur_tx = mTexFromX;

		for (int x = 0; x <= mResolution; x++)
		{
			glTexCoord2f (cur_tx, next_ty);
			glVertex3f (cur_x, next_y, 0);
			glTexCoord2f (cur_tx, cur_ty);
			glVertex3f (cur_x, cur_y, 0);

			cur_x += step_x;
			cur_tx += step_tx;
		}

		glEnd ();

		cur_y = next_y;
		cur_ty = next_ty;
	}

}

/**
 * create wire framed plane
 	*/
void cPlane::MakeWirePlane ()
{
	cerr << "MakeWirePlane not yet supported." << endl;
}
