/* (c) 1999-2000 Tino Schwarze, see COPYING for details */
/**@pkg cObject.cMaterial*/
/**
 * class for representing an OpenGL material
 *
 * #include "cMaterial.hh"
 *
 * -lMesaGL or -lGL
 *
 * A cMaterial object can be used via two ways (even in parallel):
 *
 *<ol>
 *<li>It can be given to any cObject as material to use.</li>
 *<li>It can be integrated as an cObject into the scene hierarchy
 *    to have childs.</li>
 *</ol>
 *
 * @pkgdoc cObject.cMaterial
 */

#ifndef cMaterial_hh
#define cMaterial_hh

#include <GL/gl.h>

#include "cObject.hh"
#include "cADSEColor.hh"
#include "common.hh"

/**
 * a class for representation of an OpenGL material
 	*
	* This is thought to be the base material class.
	*
	* It might be derived from it later. (e.g. for Texturing)
	*/
class cMaterial : public cObject
{
public:
// LIFETIME
	/**
	 * default constructor
	 	* @param name optional name of object
		*/
	cMaterial (const char *name = NULL);

	/**
	 * constructor with full initialization
	 	* @param color material's color (cADSEColor object)
		* @param shininess how shiny is the surface (0..128)
		* @param name optional name of object
		*/
	cMaterial (
		const cADSEColor &color = cADSEColor (),
		const int shininess = 0,
		const char *name = NULL);

	/**
	 * destructor
	 	*/
	virtual ~cMaterial ();

	/**
	 * not used, but if omitted, it might not be overloaded later
	 	*/
	virtual int Init ();

	/**
	 * start using the material's properties
	 	*
		* This is only neccessary to embed a cMaterial into a
		* cVisibleObject. We cannot use Activate() there since
		* it will call cObject::Activate() which in turn calls
		* Deactivate() so the material settings are undone before
		* the object gets drawn.
		*/
	virtual void StartMaterialUse ();

	/**
	 * end using the material's properties
	 	*/
	virtual void EndMaterialUse ();

protected:
// PROTECTED METHODS
	virtual const char *GetDefaultName () const 
	{
		return "cMaterial";
	}

	/**
	 * activate material
	 	*/
	virtual void Activate ();

	/**
	 * deactivate material
	 	*/
	virtual void Deactivate ();

// PROTECTED DATA
	/** color of material */
	cADSEColor mColor;

	/** we need to restore the material which was active before Activate() */
	cADSEColor mOldFrontColor;
	GLfloat mOldFrontShininess;

	/** we later need to restore back color too */
	cADSEColor mOldBackColor;
	GLfloat mOldBackShininess;

	/** shininess of material */
	int mShininess;
};

#endif // ifndef cMaterial_hh
