/* (c) 1999-2000 Tino Schwarze, see COPYING for details */
/**@pkg cObject.cVisibleObject.cInteractiveObject.cPlane.cHoledPlane*/
/**
 * plane with an hole derived from class cInteractiveObject
 *
 * looks like this:<PRE>
 * +------+
 * | +--+ |
 * | |  | |
 * | +--+ |
 * +------+</PRE>
 *
 * #include "cHoledPlane.hh"
 * -lglut
 *
 * @see cInteractiveObject
 * @pkgdoc cObject.cVisibleObject.cInteractiveObject.cPlane.cHoledPlane
 */

#ifndef cHoledPlane_hh
#define cHoledPlane_hh

#include "cPlane.hh"

/**
 * Simple plane object.
 	*
	* This object displays a simple plane, either solid or wireframed.
	* Further, a display list is used to draw the plane.
	* The plane starts at (-size_x/2,-size_y/2,0) and extends to
	* (size_x/2, size_y/2, 0)
	* Texture coordinates are generated also. (default: (0.0,0.0)-(1.0,1.0))
	*/
class cHoledPlane 
:	public cPlane
{
// LIFECYCLE
public:
	// enum ePlaneType is inherited (hopefully)

	/**
	 * default constructor w/ optional object name
	 	* @param name name of object (optional)
	 	*/
	cHoledPlane (
		cEventDispatcher *disp,
		const char *name = NULL);

	/**
	 * constructor w/ size and object name
	 	* @param size_x (GLdouble) x side length of plane (default: 1.0)
	 	* @param size_y (GLdouble) y side length of plane (default: 1.0)
	 	* @param name name of object (optional)
		*
		* If any of the hole percentages is &lt; 0 or &gt; 100, the plane
		* will not have a hole.
	 	*/
	cHoledPlane (
		cEventDispatcher *disp,
		GLdouble size_x, 
		GLdouble size_y, 
		const char *name = NULL);

	/**
	 * constructor w/ size, hole and object name
	 	* @param size_x (GLdouble) x side length of plane (default: 1.0)
	 	* @param size_y (GLdouble) y side length of plane (default: 1.0)
		* @param hole_start_percent_x where does the hole start in x direction?
		* @param hole_end_percent_x where does the hole end in x direction?
		* @param hole_start_percent_y where does the hole start in x direction?
		* @param hole_end_percent_y where does the hole end in x direction?
	 	* @param name name of object (optional)
		*
		* If any of the hole percentages is &lt; 0 or &gt; 100, the plane
		* will not have a hole.
	 	*/
	cHoledPlane (
		cEventDispatcher *disp,
		GLdouble size_x, 
		GLdouble size_y, 
		GLfloat hole_start_percent_x,
		GLfloat hole_end_percent_x,
		GLfloat hole_start_percent_y,
		GLfloat hole_end_percent_y,
		const char *name = NULL);

	/**
	 * default constructor
	 	* @param size_x (GLdouble) x side length of plane (default: 1.0)
	 	* @param size_y (GLdouble) y side length of plane (default: 1.0)
		* @param hole_start_percent_x where does the hole start in x direction?
		* @param hole_end_percent_x where does the hole end in x direction?
		* @param hole_start_percent_y where does the hole start in x direction?
		* @param hole_end_percent_y where does the hole end in x direction?
	 	* @param solid (tePlaneType) draw solid or wireframe plane (default: solid)
		* @param name name of object (optional)
	 	*/
	cHoledPlane (
		cEventDispatcher *disp,
		GLdouble size_x, 
		GLdouble size_y, 
		GLfloat hole_start_percent_x,
		GLfloat hole_end_percent_x,
		GLfloat hole_start_percent_y,
		GLfloat hole_end_percent_y,
		tePlaneType solid,
		const char *name = NULL);

	/**
	 * default constructor
	 	* @param size_x (GLdouble) x side length of plane (default: 1.0)
	 	* @param size_y (GLdouble) y side length of plane (default: 1.0)
		* @param hole_start_percent_x where does the hole start in x direction?
		* @param hole_end_percent_x where does the hole end in x direction?
		* @param hole_start_percent_y where does the hole start in x direction?
		* @param hole_end_percent_y where does the hole end in x direction?
	 	* @param solid (tePlaneType) draw solid or wireframe plane (default: solid)
		* @param resolution subdivisions per side
		* @param name name of object (optional)
	 	*/
	cHoledPlane (
		cEventDispatcher *disp,
		GLdouble size_x, 
		GLdouble size_y, 
		GLfloat hole_start_percent_x,
		GLfloat hole_end_percent_x,
		GLfloat hole_start_percent_y,
		GLfloat hole_end_percent_y,
		tePlaneType solid,
		int resolution,
		const char *name = NULL);

	/**
	 * destructor
	 	*/
	virtual ~cHoledPlane ();

	// Init is inherited

	// SetTextureDomain is inherited

protected:	
	// DrawThisObject is inherited

	/**
	 * return class name
	 	*/
	virtual const char *GetDefaultName () const
	{
		return "cHoledPlane";
	}

	/**
	 * create solid plane at given resolution
	 	*/
	virtual void MakeSolidPlane ();

	
	/**
	 * create wire framed plane at given resolution
	 	*/
	virtual void MakeWirePlane ();

	/** size and pos. of the hole (all in percent!) (<0 => not used) */
	GLfloat mHoleFromX, mHoleToX;
	GLfloat mHoleFromY, mHoleToY;
};

#endif	// ifndef cHoledPlane_hh
