/* (c) 1999-2000 Tino Schwarze, see COPYING for details */
/**@pkg cObject.cMaterial.cTextureMaterial.cEnvTextureMaterial*/
/**
 * class for representing an OpenGL 2D textured material
 * which provides environment mapping
 *
 * #include "cEnvTextureMaterial.hh"
 *
 * -lMesaGL or -lGL
 *
 * A cTextureMaterial object can be used the same way as a
 * cMaterial object.
 * 
 * 1D textures are not supported.
 *
 * @see cMaterial
 *
 * @pkgdoc cObject.cMaterial.cTextureMaterial.cEnvTextureMaterial
 */

#ifndef cEnvTextureMaterial_hh
#define cEnvTextureMaterial_hh

#include <GL/gl.h>

#include "cTextureMaterial.hh"
#include "common.hh"

/**
 * a class for representation of an OpenGL texture material with
 * environment mapping
 	*
	* It's only slightly different to cTextureMaterial.
	*/
class cEnvTextureMaterial 
:	public cTextureMaterial
{
public:
// LIFETIME
	/**
	 * default constructor
	 	* @param name optional name of object
		*/
	cEnvTextureMaterial (const char *name = NULL);

	/**
	 * constructor with initialization
	 	* @param color material's color (cADSEColor object)
		* @param shininess how shiny is the surface (0..128)
		* @param name optional name of object
		*/
	cEnvTextureMaterial (
		const cADSEColor &color = cADSEColor (),
		const int shininess = 0,
		const char *name = NULL);

	/**
	 * constructor with initialization
		* @param texture file name of texture to load
		* The texture is loaded by Init(). Only JPEG files are supported.
	 	* @param color material's color (cADSEColor object)
		* @param shininess how shiny is the surface (0..128)
		* @param name optional name of object
		*/
	cEnvTextureMaterial (
		const char *texture,
		const cADSEColor &color,
		const int shininess,
		const char *name = NULL);

	// if constructors could be inherited...

	/**
	 * constructor with initialization
		* @param texture file name of texture to load
		* The texture is loaded by Init(). Only JPEG files are supported.
		* @param usemipmaps enable mipmapping?
	 	* @param color material's color (cADSEColor object)
		* @param shininess how shiny is the surface (0..128)
		* @param name optional name of object
		*/
	cEnvTextureMaterial (
		const char *texture,
		bool usemipmaps,
		const cADSEColor &color,
		const int shininess,
		const char *name = NULL);

	/**
	 * copy constructor - important!
	 	*/
	cEnvTextureMaterial (const cEnvTextureMaterial &ctm);

	/**
	 * destructor
	 	*/
	virtual ~cEnvTextureMaterial ();

	/**
	 * initialization: loads texture
	 	*/
	virtual int Init ();

	/* we inherit: SetMipmapping, IsMipmapped */

	/**
	 * start using the material's properties
	 	*
		* This is only neccessary to embed a cMaterial into a
		* cVisibleObject. We cannot use Activate() there since
		* it will call cObject::Activate() which in turn calls
		* Deactivate() so the material settings are undone before
		* the object gets drawn.
		*/
	virtual void StartMaterialUse ();

	/**
	 * end using the material's properties
	 	*/
	virtual void EndMaterialUse ();

	/* we inherit:
	 * - LoadTexture
	 * - UseTexture
	 * - SetTextureParameters
	 */
protected:
// PROTECTED METHODS
	virtual const char *GetDefaultName () const 
	{
		return "cEnvTextureMaterial";
	}

	/* we inherit:
	 * - UpdateTexture
	 * - Activate
	 * - Deactivate
	 * - AllocateTextureName
	 */

// PROTECTED DATA
	// we need to save state between StartMaterialUse() and
	// EndMaterialUse ()
	GLint mOldSMode, mOldTMode;
	GLboolean mOldSState, mOldTState;
};

#endif // ifndef cEnvTextureMaterial_hh
